#include <stdlib.h>
#include "string.h"
#include "AudioStack/clStack.h"

#include "AudioStack/SMT/clSrcState.h"
#include "AudioStack/AudioSources/clAudioSourceController.h"
#include "AudioStack/clAudioSMEngine.h"
#include "AudioStack/AudioSources/clFactory_AudioSourceClass.h"

#ifndef USE_DLT_TRACE
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_COMP_AUDIOSTACK
#include "trcGenProj/Header/clStack.cpp.trc.h"
#endif

namespace AudioStack
{

using namespace AudioSource;

clStack::clStack()
{
   OSAL_pvMemorySet(m_sources, 0, sizeof(m_sources));
   m_SinkID = 1;
   m_SP = 0;
   name = "AST";
}

clStack::clStack(int sinkID, const char* myName)
{
   OSAL_pvMemorySet(m_sources, 0, sizeof(m_sources));
   m_SP = 0;
   m_SinkID = sinkID;
   name = std::string(myName);
   //ToDO How can we assign the stack names? From xml ?? Find a way to set the number of possible sinks in the system

}

clStack::~clStack()
{
  OSAL_pvMemorySet(m_sources, 0, sizeof(m_sources));
  m_SP = 0;
}

void clStack::vResetStack()
{
   OSAL_pvMemorySet(m_sources, 0, sizeof(m_sources));
   m_SP = 0;
}

int clStack::GetSinkID()
{
   return m_SinkID;
}

clAudioSource* clStack::GetTopSource(sourceClassID srcClass) const
{

   for(int SP=m_SP; SP>0; SP--) {
    if( m_sources[SP-1] != NULL){
      if(m_sources[SP-1]->sGetId().enSourceClass ==  srcClass)
      {
       return m_sources[SP-1];
      }
    }
   }
   return NULL;
}

clAudioSource* clStack::GetTopSource_Type(sourceTypeID srcTypeID) const
{

   for(int SP=m_SP; SP>0; SP--) {
     if( m_sources[SP-1] != NULL){
      if(m_sources[SP-1]->getSourceType().id == srcTypeID)
      {
       return m_sources[SP-1];
      }
     }
   }
   return NULL;
}

/*
 * Function check if 'srcAllowedOnTopOf' is aborted/removed from stack after applying
 * all abort rules configured for 'src', if 'srcAllowedOnTopOf' is aborted then return true
 * else return false
 * src          : source for which abort() rules are getting applied
 * srcAllowedOnTopOf : source class ID of source above which 'src' is allowed
 * Ex : MUTE_SYSTEM(on)/MUTE_ENT(pause)/MEDIA#1(pasue)
 *     In this case MEDIA is allowed on the top of MUTE_ENT and Abort(MUTE_ENT) also present in rule
 *     After applying abort rules MUTE_ENT is removed from stack, hence function return true
 */
bool clStack::DoPushActions(clAudioSource &src,sourceClassID srcAllowedOnTopOf)
{

   if((src.bIsSourceBG() == true) && (clFactory_AudioSourceClass::GetSourceSettings().BackGroundSourceExchange == true))
   {
    ETG_TRACE_ERR(("DoPushActions, background source exchange is undergoing, No need to handle Push Action "));
    return false;
   }
   bool bSrcRemoved = false;
   clSourceClass const & srcClass = clFactory_AudioSourceClass::GetSourceClass(src.sGetId().enSourceClass);
   for(std::vector<clSourceClass::actions_t>::const_iterator it = srcClass.PushActions.begin();
         it != srcClass.PushActions.end();
         ++it)
   {
      switch((*it) & clSourceClass::/*actions_t::*/mask) {
         case clSourceClass::/*actions_t::*/action_abort:
         {
            ETG_TRACE_USR3(("Found Abort action: aborting SourceClass: %d",
                  ((*it) & clSourceClass::/*actions_t::*/idxMask)));
            RemoveInternal_SourceClass(
                  tU8(((*it) & clSourceClass::/*actions_t::*/idxMask)), src.sGetId());

            /*
             * if below source which allowed new source is aborted
             */
            if(srcAllowedOnTopOf == ((*it) & clSourceClass::/*actions_t::*/idxMask))
            {
              bSrcRemoved = true;
            }


         }break;
         default:
            ETG_TRACE_ERR(("Push Action not handled"));
            break;
      }
   }
   return bSrcRemoved;
   /* loop over possible actions */
   //  for(i=0; (action=src.getPushAction(i)) != clSourceClass::/*actions_t::*/end; i++) {
   //
   //    switch(action & clStackRules::/*actions_t::*/mask) {
   //    case clStackRules::/*actions_t::*/action_abort:
   //         ETG_TRACE_USR4(("Found Abort action for: %d",
   //                               ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(action & clStackRules::/*actions_t::*/idxMask))));
   //         RemoveInternal_SourceClass(
   //               (action & clStackRules::/*actions_t::*/idxMask), src.sGetId());
   //      break;
   //    }
   //  }
}

void clStack::DoPopActions(clAudioSource &src)
{
   if((src.bIsSourceBG() == true) && (clFactory_AudioSourceClass::GetSourceSettings().BackGroundSourceExchange == true))
   {
    ETG_TRACE_ERR(("DoPushActions, background source exchange is undergoing, No need to handle Push Action "));
    return;
   }
   clSourceClass const & srcClass = clFactory_AudioSourceClass::GetSourceClass(src.sGetId().enSourceClass);
   for(std::vector<clSourceClass::actions_t>::const_iterator it = srcClass.PopActions.begin();
         it != srcClass.PopActions.end();
         ++it)
   {
      switch((*it) & clSourceClass::/*actions_t::*/mask) {
         case clSourceClass::/*actions_t::*/action_abort:
         {
            ETG_TRACE_USR3(("Found Abort action: aborting SourceClass: %d",
                  ((*it) & clSourceClass::/*actions_t::*/idxMask)));
            RemoveInternal_SourceClass(
                  tU8(((*it) & clSourceClass::/*actions_t::*/idxMask)), src.sGetId());
         }break;
         default:
            ETG_TRACE_ERR(("Push Action not handled"));
            break;
      }
   }
}
/*
 * vdu1kor (Fix NCG3D-10801)
 * this function is called if source is already on stack then it check if
 * source which is already on the stack is allowed on the top of sources
 * which are present above them. if allowed then only pushaction is performed
 * so that if abort(Mute_Ent) is present then should be removed from stack.
 * Ex :
 * AST:  --------TOP--------
 * AST: MUTE_ENTERTAIN  : on
 * AST: SPI_ENTERTAIN#0 : pause
 * AST:  ------BOTTOM-------
 * If setRouteRequest(SPI,on) is received then if SPI_ENT is allowed over Mute_ent then
 * it check if also abort(mute_ent) rule present then it remove mute_ent.
 *
 */
bool clStack::PushActionForSourceAlreadyOnStack(clAudioSource &src,uint8_t indexSrcOnStack)
{
  clAudioSource* pStackSrc = NULL;
  int SP;

  const stSourceSetting& srcSettings = clFactory_AudioSourceClass::GetSourceSettings();
  if(!srcSettings.AllowSrcAlreadyonStack)
  {
    ETG_TRACE_ERR(("PushActionForSourceAlreadyOnStack, source already on stack, No need to apply rules !!!"));
    return false;
  }
  ETG_TRACE_ERR(("PushActionForSourceAlreadyOnStack, source already on stack, apllying stack rules !!"));
  //until there is a source to check
  for(SP=m_SP; SP>0; SP--)
  {
  pStackSrc = m_sources[SP-1];
    SourceID StackSrcID = pStackSrc->sGetId();
    const clSourceClass & StackSrcClass = clFactory_AudioSourceClass::GetSourceClass(StackSrcID.enSourceClass);

    //if(StackSrcID == src.sGetId())
    //{
    //  /*
    //   * If source is same, no need to check if FM is allowed on the top of FM
    //   * To Avoid scenario where source is not allowed but abort rule present
    //   * Ex : MUTE_SYSTEM(on)/FM(pause) => request for FM(on)
    //   *      In this case Abort(Mute_System) is present in FM ruleset but
    //   *      FM not allowed over MuteSystem [one_source_two_mutes.xml]
    //   *
    //   */
    //  return false;
    //}

    clSourceClass::allow_t allow = StackSrcClass.allowOnTop(src.sGetId());
    ETG_TRACE_USR4(("AddSource: Allowance check ClassID %d SubID %d: %d"
        , StackSrcID.enSourceClass
        , StackSrcID.u16SubSource
        , allow));
    //check if it's allowed
    if(allow == clSourceClass::allowed)
    {
      if(false == DoPushActions(src,StackSrcID.enSourceClass))
      {
        ETG_TRACE_USR2(("AddSource: stackable Source ClassID %d SubID %d is not Top of source"
            , src.sGetId().enSourceClass
            , src.sGetId().u16SubSource));

        /*
         *  Use case 1: MUTE_AUDIO_OFF(on)/TUNER_TA_FM(on)/SDS_SPEECHRECOGNITION(pause)/Media(pause)
         *  Operation: User trigger Source Change for SDS_SPEECH again,
         *  Precondition: SDS_SPEECH is allowed on the top of TA but not allowed over MUTE_AUDIO_OFF,
         *          SDS_SPEECH not allowed over MUTE_AUDIO_OFF
         *  DoPushActions(), returned false as TA is not Aborted, index of SDS_SPEECH(1) is less than index of TA(2)
         *  Swap is needed
         */

        if(indexSrcOnStack<SP-1){

            ETG_TRACE_USR3(("PushActionForSourceAlreadyOnStack: stackable Source [ClassID %d SubID %d] is not  \
                         Top of source [ClassID %d SubID %d], Swapping sources !"
                         , src.sGetId().enSourceClass, src.sGetId().u16SubSource
                         , StackSrcID.enSourceClass, StackSrcID.enSourceClass));

        //Exchange of source is required
        m_sources[indexSrcOnStack] = m_sources[SP-1];
        m_sources[SP-1] = &src;
        }
        else
        {
            /*
             *  Use case 2: phone(on)/Mute_Ent(off)/Media(pause) [PSARCCB-9948]
             *  User trigger Ent_Mute again, Mute_Ent is allowed on the top of Media but
             *  as Abort(Media) is not present
             *  after DoPushActions(), index of Mute_Ent (1) is greater than index of Media(0)
             *  No need to swap the sources.
             */
          ETG_TRACE_USR4(("PushActionForSourceAlreadyOnStack: No Need to swap"));
        }
      }
      else
      {
        /*
         *  Use case 3: Mute_System(on)/Mute_Ent(off)/Media(pause)
         *  User trigger Source Change for Media again, Media is allowed on the top of Mute_Ent &
         *  Abort(Mute_Ent) is also present
         *  DoPushActions(), returned true as Mute_Ent is Aborted
         */
        ETG_TRACE_USR3(("PushActionForSourceAlreadyOnStack: stackable Source [ClassID %d SubID %d] aborted source [ClassID %d SubID %d]"
            , src.sGetId().enSourceClass, src.sGetId().u16SubSource
            , StackSrcID.enSourceClass, StackSrcID.enSourceClass));
      }

      return true;
    }
    //check if it's kicked (paj5kor)
    if(allow == clSourceClass::kicked)
    {
      ETG_TRACE_USR3(("AddSource: stackable Source ClassID %d SubID %d PROHIBITS ADDING SOURCE"
          , StackSrcID.enSourceClass
          , StackSrcID.u16SubSource));
      //do not add and exit
      return false;
    }
    //check next below
    //pStackSrc = GetUnderlayingSource(*pStackSrc);
  }//End of for loop
  return true;
}
bool clStack::AddSource(clAudioSource &src)
{
   /* check stack pointer */
   if (m_SP == (MAX_AUDIO_SOURCES-1))
   {
      ETG_TRACE_ERR(("AddSource: ERROR: STACK ALREADY MAX (%d) SOURCES"
            , MAX_AUDIO_SOURCES));
      return false;
   }

   /* add first element on stack */
   if (m_SP == 0) {
      ETG_TRACE_USR4(("AddSource: add first source"));
      m_sources[m_SP++] = &src;
      //add this sink to AudioSource
      src.addSink(m_SinkID);
      return true;
   }

   //Get Info about incoming source
   SourceID srcID(src.sGetId());
   const clSourceClass& srcClass  = clFactory_AudioSourceClass::GetSourceClass(srcID.enSourceClass);
   clAudioSource* pStackSrc = GetSourceInStack_EqualSourceType(srcID.enSourceClass);
   ETG_TRACE_USR4(("AddSource: ClassID: %d, SubID: %d, stackable: %d"
         , srcID.enSourceClass
         , srcID.u16SubSource
         , clFactory_AudioSourceClass::GetType(srcID).stackable ));
   int SP;
   //Check if source already in Stack
   for(SP=m_SP; SP>0; SP--) {

    if(m_sources[SP-1]->sGetId() == srcID)
      {
         //if it is not stackable
         if(!clFactory_AudioSourceClass::GetType(srcID).stackable)
            // not stackable source (e.g. background), which is still on stack has to be shifted to top of bg sources
         {
      if ((SP<m_SP) && (clFactory_AudioSourceClass::GetTypeID(m_sources[SP]->sGetId()) == srcClass.getTypeID()))
      {
         ETG_TRACE_USR4(("AddSource: shifting on top of ClassID: %d SubID: %d"
           , m_sources[SP]->sGetId().enSourceClass
           , m_sources[SP]->sGetId().u16SubSource ));
         // there is another bg-source above of this, so put this one up
         m_sources[SP-1] = m_sources[SP];
         //daw2hi add sink to source
         src.addSink(m_SinkID);
         m_sources[SP] = &src;
      }else{
         ETG_TRACE_USR4(("AddSource: not stackable source not shifted."));
          PushActionForSourceAlreadyOnStack(src,uint8_t(SP-1));
      }
         }
      else
      {
        ETG_TRACE_USR4(("AddSource: stackable source, already on stack."));
        PushActionForSourceAlreadyOnStack(src,uint8_t(SP-1));

      }
         // no action for fg source
         return true;
      }
   }

   //Check if there is Entertainment source in stack
   // AND given source is Entertainment

   if(clFactory_AudioSourceClass::GetType(srcID).stackable)
   {
      //place it on top of first allowed
      pStackSrc = GetTopSource();
      //until there is a source to check
      while(pStackSrc != NULL)
      {
         SourceID StackSrcID = pStackSrc->sGetId();
         const clSourceClass & StackSrcClass = clFactory_AudioSourceClass::GetSourceClass(StackSrcID.enSourceClass);
         clSourceClass::allow_t allow = StackSrcClass.allowOnTop(src.sGetId());
         ETG_TRACE_USR4(("AddSource: Allowance check ClassID %d SubID %d: %d"
               , StackSrcID.enSourceClass
               , StackSrcID.u16SubSource
               , allow));
         //check if it's allowed
         if(allow == clSourceClass::allowed)
         {
            //insert on top of that source
            int SP2;
            for(SP2=m_SP;
                SP2>0
                && m_sources[SP2-1]->sGetId() != StackSrcID;
                SP2--)
            {
               m_sources[SP2] = m_sources[SP2-1];
            }
            //daw2hi announce sink  to source
            src.addSink(m_SinkID);
            m_sources[SP2] = &src;
            //stack is now one element longer
            m_SP++;
            //apply actions to the stack
            DoPushActions(src);
            ETG_TRACE_USR3(("AddSource: Source added on of top ClassID %d SubID %d"
                  , StackSrcID.enSourceClass
                  , StackSrcID.u16SubSource));
            return true;
         }
         //check if it's kicked (paj5kor)
         if(allow == clSourceClass::kicked)
         {
            ETG_TRACE_USR3(("AddSource: stackable Source ClassID %d SubID %d PROHIBITS ADDING SOURCE"
                  , StackSrcID.enSourceClass
                  , StackSrcID.u16SubSource));
            //do not add and exit
            return false;
         }
         //check next below
         pStackSrc = GetUnderlayingSource(*pStackSrc);
      }

   }else{
    //daw2hi: without this, it is always NULL
    pStackSrc = GetTopSource();
      //is there already same type in stack?
      if(pStackSrc != NULL)
      {
         //ask each source from top till same type if someone denies

         for(SP=m_SP; SP>0; SP--) {
            //only place it on top of same type for replacement
            if(clFactory_AudioSourceClass::GetTypeID(m_sources[SP-1]->sGetId()) == srcClass.getTypeID())
            {
               int SP2;
               for(SP2=m_SP; SP2>SP; SP2--) {
                  m_sources[SP2] = m_sources[SP2-1];
               }
               //daw2hi add sink to source
               src.addSink(m_SinkID);
               m_sources[SP] = &src;
               /* stack is now one element longer */
               m_SP++;
               /* apply actions to the stack */
               DoPushActions(src);
               /* successful */
               return true;
            }else{
               //   do not add if denied before
               const clSourceClass & Src_Class = clFactory_AudioSourceClass::GetSourceClass(m_sources[SP-1]->sGetId().enSourceClass);
               clSourceClass::allow_t allow_source = Src_Class.allowOnTop(src.sGetId());
               if(allow_source == clSourceClass::kicked)
               {
                  ETG_TRACE_USR3(("AddSource: not stackable Source ClassID %d SubID %d PROHIBITS ADDING SOURCE"
                        , m_sources[SP-1]->sGetId().enSourceClass
                        , m_sources[SP-1]->sGetId().u16SubSource));
                  return false;
               }
            }
         }
      }else{
         //there's no source of same type in stack
         //go from top till end:
         for(SP=m_SP; SP>0; SP--) {
            SourceID SrcID = m_sources[SP-1]->sGetId();

      //VVD lint fix
            const clSourceClass & src_Class = clFactory_AudioSourceClass::GetSourceClass(SrcID.enSourceClass);

      //VVD lint fix
      clSourceClass::allow_t allowSrc = src_Class.allowOnTop(src.sGetId());
            switch(allowSrc) {
               case clSourceClass::/*allow_t::*/allowed:
                  //   if someone allows insert add it on top
                  /* insert the new source above the current one */
                  int SP2;
                  for(SP2=m_SP; SP2>SP; SP2--) {
                     m_sources[SP2] = m_sources[SP2-1];
                  }
                  //daw2hi add sink to source
                  src.addSink(m_SinkID);
                  m_sources[SP] = &src;
                  /* stack is now one element longer */
                  m_SP++;
                  /* apply actions to the stack */
                  DoPushActions(src);
                  /* successful */
                  return true;
               case clSourceClass::/*allow_t::*/kicked:
               {
                  //   if someone deny do not add
                  ETG_TRACE_USR3(("AddSource: KICK RULE: SourceClass %d will prevent insert "
                        , SrcID.enSourceClass));
                  return false;
               }
               default:
                  break;
            }
         }
      }
   }
   //if end is reached insert there
   /* insert the new source above the current one */
   //int SP2;
   for(SP=m_SP; SP>0; SP--) {
      m_sources[SP] = m_sources[SP-1];
   }
   //daw2hi add sink to source
   src.addSink(m_SinkID);
   m_sources[SP] = &src;
   /* stack is now one element longer */
   m_SP++;
   /* apply actions to the stack */
   DoPushActions(src);
   /* successful */
   return true;
}


clAudioSource* clStack::AllowSource(clAudioSource &newSrc)
{
  /*
   * vdu1kor 18/03/2016
   * Condition to check if background source change is undergoing
   * if so then no need to check for allowance rule
   * If background source is denied then it may end up in situation where no source is active
   * Ex : PHONE(off)/MEDIA_PLAYER(SrcAvailCheck_P)
   *       case1> if usb device is removed then background source change is triggered
   *       case2> if on startup, availability is not received within defined time then
   *           background source change is triggered (NCG3D-11300)
   * In both use case, MediaPlayer source should be replaced with FM(pause)
   * Ex : PHONE(on)/FM(pause)
   *     case1> if Usb is inserted, then background src exchange is triggered but MEDIA
   *         is not allowed on the top of Phone hence FM will not be replaced by Media
   */

  if((true == newSrc.bIsSourceBG()) && (true == newSrc.bIsPrevSourceRemoved()))
  {
    ETG_TRACE_USR3(("AllowSource: New source is BackGround Source Exchange due to sourceRemoved, No need to check allowance, SourceClass: %d subID %d"
        , (newSrc.sGetId()).enSourceClass
        , (newSrc.sGetId()).u16SubSource));

    return &newSrc;
  }
   SourceID newSrcID = newSrc.sGetId();
   ETG_TRACE_USR3(("AllowSource: check if SourceClass %d, SubID %d is allowed"
         ,newSrcID.enSourceClass
         ,newSrcID.u16SubSource));
   /* check stack pointer */
   if (m_SP == (MAX_AUDIO_SOURCES-1))
   {
      ETG_TRACE_USR4(("AllowSource: ERROR MAX SOURCES ON STACK! "));
      return NULL;
   }
   /* add first element on stack */
   if (m_SP == 0) {
      ETG_TRACE_USR4(("AllowSource: Stack empty, insert first source "));
      return &newSrc;
   /* add new element, ask the stack for the position */
   } else {
      int SP;
      for(SP=m_SP; SP>0; SP--) {
         /* check if source already/still on stack */
       if(m_sources[SP-1] == NULL)
       {
         ETG_TRACE_USR4(("AllowSource: m_sources[SP-1] is NULL pointer "));
         return NULL;
       }
       if(m_sources[SP-1]->sGetId() == newSrcID)
         {
            ETG_TRACE_USR4(("AllowSource: Allowed, source already in Stack "));
            return &newSrc;
         }
         //Check if some source denies
         sourceClassID classID = m_sources[SP-1]->sGetId().enSourceClass;
         const clSourceClass& checkSrcClass = clFactory_AudioSourceClass::GetSourceClass(classID);

         clSourceClass::allow_t allow = checkSrcClass.allowOnTop(newSrcID);
         switch(allow) {
            case clSourceClass::/*allow_t::*/allowed:
               /* successfull */
               return m_sources[SP-1];
            case clSourceClass::/*allow_t::*/kicked:
            {
               return NULL;
            }
            default:
               break;
         }
      }
      //no one allowed before, no one kicked
      //then it is allowed
      return &newSrc;
   }
}

bool clStack::RemoveInternal_SourceClass(sourceClassID rmSrcClass, SourceID nextSrcID)
{
   ETG_TRACE_USR3(("RemoveInternal_SourceClass: Force Off for SourceClass: %d"
            ,ETG_CENUM(AudioSources::enAudioSources,rmSrcClass)));
   clAudioSource* pRmSrc = GetTopSource(rmSrcClass);
   while(pRmSrc != NULL)
   {
      ETG_TRACE_USR1(("RemoveInternal_SourceClass: Check SourceClass: %d subID %d"
         , pRmSrc->sGetId().enSourceClass
         , (tU16)pRmSrc->sGetId().u16SubSource));
      //RemoveSrcInternal may remove Source completely synchronously
      //so we need to remeber which source was below
      clAudioSource* pUnderlayingSrc = GetUnderlayingSource(*pRmSrc);
      if(pRmSrc->sGetId().enSourceClass == rmSrcClass)
      {
         return RemoveInternal_Source(pRmSrc->sGetId(),nextSrcID);
      }
      pRmSrc = pUnderlayingSrc;
   }
   return false;
}

bool clStack::RemoveInternal_Source(SourceID rmSrcID, SourceID nextSrcID)
/*
 internal remove function. calls OFF of AudioSource Controller ind indirect the public remove function
 */
{
  (void) nextSrcID;
  int SP;
  clAudioSource *src;

  /* add to stack, ask element on top of stack */
  for(SP=m_SP-1; SP>=0; SP--) {
    if (m_sources[SP]->sGetId() == rmSrcID) {

      /* remember this element */
      src = m_sources[SP];

      /* switch off source */
      //OFF(src, u8NextSrcId);
      ETG_TRACE_USR1(("RemoveSourceInternal: Force Off for SourceClass: %d subID %d"
         , src->sGetId().enSourceClass
         , (tU16)src->sGetId().u16SubSource));

      clAudioSMEngine::Source_Off(src->sGetId(), static_cast<tU32>(BITMASK_FORCE_OFF));
      /* do pop actions */
      DoPopActions(*src);

      // direct and synchronous removal or src, if already in state off
      // otherwise removal will be done on: Off_Done()
      if (src->bIsOff())
      {
        RemoveSource(*src);
      }
      /* element popped */
      return true;
    }
  }
   ETG_TRACE_USR4(("RemoveSourceInternal: Not active: SourceClass: %d subID %d"
         , rmSrcID.enSourceClass
         , (tU16)rmSrcID.u16SubSource));
  /* source not popped */
  return false;
}

bool clStack::RemoveSource(clAudioSource &src)
{
   int SP;

   /* remove from stack, ask element on top of stack */
   for(SP=m_SP-1; SP>=0; SP--) {
      if (m_sources[SP] == &src) {
         for(; SP<m_SP-1; SP++) {   //lint !e445 PQM_authorized_246. Reason: Same loop variable used intentionally, reviewed and not critical
            if (SP>=0){ // for LINT
               m_sources[SP] = m_sources[SP+1];
            }
         }
         m_SP--;

         /* apply pop actions to the stack */
         DoPopActions(src);

         /* element popped */
         return true;
      }
   }

   /* source not popped */
   return false;
}

clAudioSource * clStack::GetUnderlayingSource(clAudioSource &src) const
{
  int SP;

  /* ... */
  for(SP=m_SP-1; SP>=0; SP--) {
    if (m_sources[SP] == &src) {
      if (SP == 0) return NULL;
      else return m_sources[SP-1];
    }
  }

  return NULL;
}


clAudioSource * clStack::GetTopSource() const
{
  if (m_SP == 0) return NULL;
  else return m_sources[m_SP-1];
}

void clStack::trace() const
{
  int SP;

  ETG_TRACE_USR1(("%s:  --------TOP--------",name.c_str()));
  char oneSource[250];

  for(SP=m_SP-1; SP>=0; SP--)
  {
    if(m_sources[SP] != NULL)
    {
    snprintf(oneSource,250,"%5s:  %15s : %15s (SrcClass %d(%#x), SubID %d(%#x))"
          ,name.c_str()
            ,m_sources[SP]->pacGetName()
            ,m_sources[SP]->pclGetState((tU16)m_SinkID)->m_pacName
            ,m_sources[SP]->sGetId().enSourceClass
            ,m_sources[SP]->sGetId().enSourceClass
            ,m_sources[SP]->sGetId().u16SubSource
            ,m_sources[SP]->sGetId().u16SubSource);

      ETG_TRACE_USR1(("%s",oneSource));
    }
  }
  ETG_TRACE_USR1(("%s:  ------BOTTOM-------",name.c_str()));
}

bool clStack::OFF(clAudioSource *source, SourceID srcID)
{
  if(source != NULL)
  {
    source->vOff(srcID);
    return TRUE;
  }
  return FALSE;
}


clAudioSource *clStack::GetSourceInStack_EqualSourceType(clSourceClass::sourceTypeID typeID) const
{
  int SP;

  // start at top of stack
  for(SP=m_SP-1; SP>=0; SP--) {
    if(m_sources[SP] != NULL)
    {
      sourceClassID checkSrc = m_sources[SP]->sGetId().enSourceClass;
      const clSourceClass& checkSrcClass = clFactory_AudioSourceClass::GetSourceClass(checkSrc);
      if (checkSrcClass.SourceTypeID == typeID) {
        return m_sources[SP];
      }
    }
  }
  return NULL;
}


}//namespace

/* EOF */
